<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\EducationModel;

class EducationController extends BaseController
{
    protected $educationModel;

    public function __construct()
    {
        $this->educationModel = new EducationModel();
    }

    public function index()
    {
        $search = $this->request->getGet('search') ?? '';
        $filter = [
            'jenjang' => $this->request->getGet('jenjang') ?? '',
            'tahun_mulai' => $this->request->getGet('tahun_mulai') ?? '',
            'tahun_selesai' => $this->request->getGet('tahun_selesai') ?? '',
            'status_pendidikan' => $this->request->getGet('status_pendidikan') ?? ''
        ];
        $sort = $this->request->getGet('sort') ?? 'urutan';
        $order = $this->request->getGet('order') ?? 'ASC';
        $page = (int)($this->request->getGet('page') ?? 1);

        $builder = $this->educationModel->searchAndFilter($search, $filter, $sort, $order);
        $total = $builder->countAllResults(false);
        
        $education = $builder->get(10, ($page - 1) * 10)->getResultArray();

        $pager = \Config\Services::pager();
        $pager->store('education', $page, 10, $total);

        $data = [
            'title' => 'Kelola Riwayat Pendidikan',
            'education' => $education,
            'pager' => $pager,
            'search' => $search,
            'filter' => $filter,
            'sort' => $sort,
            'order' => $order
        ];

        return view('admin/education/index', $data);
    }

    public function create()
    {
        $data = [
            'title' => 'Tambah Riwayat Pendidikan'
        ];
        return view('admin/education/create', $data);
    }

    public function store()
    {
        $rules = [
            'jenjang' => 'required|in_list[SD,SMP,SMA,SMK,D3,S1,S2,S3]',
            'nama_sekolah' => 'required|min_length[3]|max_length[200]',
            'status_pendidikan' => 'required|in_list[lulus,sedang_menempuh,tidak_lulus,cuti,pindah]'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $data = [
            'jenjang' => $this->request->getPost('jenjang'),
            'nama_sekolah' => $this->request->getPost('nama_sekolah'),
            'jurusan' => $this->request->getPost('jurusan'),
            'tahun_mulai' => $this->request->getPost('tahun_mulai'),
            'tahun_selesai' => $this->request->getPost('tahun_selesai'),
            'nilai_akhir' => $this->request->getPost('nilai_akhir'),
            'deskripsi' => $this->request->getPost('deskripsi'),
            'status_pendidikan' => $this->request->getPost('status_pendidikan'),
            'urutan' => (int)($this->request->getPost('urutan') ?? 0)
        ];

        if ($this->educationModel->insert($data)) {
            return redirect()->to('admin/education')->with('success', 'Riwayat pendidikan berhasil ditambahkan!');
        } else {
            return redirect()->back()->withInput()->with('error', 'Gagal menambahkan riwayat pendidikan!');
        }
    }

    public function edit($id)
    {
        $education = $this->educationModel->find($id);
        
        if (!$education) {
            return redirect()->to('admin/education')->with('error', 'Riwayat pendidikan tidak ditemukan!');
        }

        $data = [
            'title' => 'Edit Riwayat Pendidikan',
            'education' => $education
        ];
        return view('admin/education/edit', $data);
    }

    public function update($id)
    {
        $education = $this->educationModel->find($id);
        
        if (!$education) {
            return redirect()->to('admin/education')->with('error', 'Riwayat pendidikan tidak ditemukan!');
        }

        $rules = [
            'jenjang' => 'required|in_list[SD,SMP,SMA,SMK,D3,S1,S2,S3]',
            'nama_sekolah' => 'required|min_length[3]|max_length[200]',
            'status_pendidikan' => 'required|in_list[lulus,sedang_menempuh,tidak_lulus,cuti,pindah]'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $data = [
            'jenjang' => $this->request->getPost('jenjang'),
            'nama_sekolah' => $this->request->getPost('nama_sekolah'),
            'jurusan' => $this->request->getPost('jurusan'),
            'tahun_mulai' => $this->request->getPost('tahun_mulai'),
            'tahun_selesai' => $this->request->getPost('tahun_selesai'),
            'nilai_akhir' => $this->request->getPost('nilai_akhir'),
            'deskripsi' => $this->request->getPost('deskripsi'),
            'status_pendidikan' => $this->request->getPost('status_pendidikan'),
            'urutan' => (int)($this->request->getPost('urutan') ?? 0)
        ];

        if ($this->educationModel->update($id, $data)) {
            return redirect()->to('admin/education')->with('success', 'Riwayat pendidikan berhasil diupdate!');
        } else {
            return redirect()->back()->withInput()->with('error', 'Gagal mengupdate riwayat pendidikan!');
        }
    }

    public function delete($id)
    {
        $education = $this->educationModel->find($id);
        
        if (!$education) {
            return redirect()->to('admin/education')->with('error', 'Riwayat pendidikan tidak ditemukan!');
        }

        if ($this->educationModel->delete($id)) {
            return redirect()->to('admin/education')->with('success', 'Riwayat pendidikan berhasil dihapus!');
        } else {
            return redirect()->to('admin/education')->with('error', 'Gagal menghapus riwayat pendidikan!');
        }
    }
}

